import { NextResponse } from "next/server";
import type { NextRequest } from "next/server";
import { locales, defaultLocale } from "./lib/i18n";

function getLocale(request: NextRequest): string {
  // Simple locale detection based on accept-language or just default
  // In a real app we'd parsing the header more robustly
  const acceptLanguage = request.headers.get("accept-language");
  if (acceptLanguage) {
      // Very basic check, prefer library like @formatjs/intl-localematcher
      if (acceptLanguage.includes('ru')) return 'ru';
      if (acceptLanguage.includes('uz')) return 'uz';
  }
  return defaultLocale;
}

export function middleware(request: NextRequest) {
  // Check if there is any supported locale in the pathname
  const { pathname } = request.nextUrl;
  
  // Exclude static files and api
  if (
      pathname.startsWith('/_next') ||
      pathname.startsWith('/api') ||
      pathname.match(/\.(ico|png|jpg|jpeg|svg|gif|webp|woff|woff2|ttf|eot|css|js)$/)
  ) {
      return;
  }

  const pathnameHasLocale = locales.some(
    (locale) => pathname.startsWith(`/${locale}/`) || pathname === `/${locale}`
  );

  if (pathnameHasLocale) return;

  // Redirect if there is no locale
  const locale = getLocale(request);
  request.nextUrl.pathname = `/${locale}${pathname}`;
  // e.g. incoming request is /products
  // The new URL is now /en/products
  return NextResponse.redirect(request.nextUrl);
}

export const config = {
  matcher: [
    // Skip all internal paths (_next), static files, and api routes
    '/((?!_next|api|.*\\.).*)',
  ],
};
